/*
 * TimeSpeedChanger.java
 *
 * User: Carsten
 * Date: 26.04.2017
 */

package carsten.risingworld.tsc;

import net.risingworld.api.Plugin;
import net.risingworld.api.Server;
import net.risingworld.api.Timer;
import net.risingworld.api.events.EventMethod;
import net.risingworld.api.events.Listener;
import net.risingworld.api.events.Threading;
import net.risingworld.api.events.player.PlayerCommandEvent;
import net.risingworld.api.events.player.PlayerSleepEvent;
import net.risingworld.api.events.player.PlayerSpawnEvent;
import net.risingworld.api.gui.Font;
import net.risingworld.api.gui.GuiLabel;
import net.risingworld.api.gui.PivotPosition;
import net.risingworld.api.objects.Player;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.FileSystems;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

public class TimeSpeedChanger extends Plugin implements Listener
{
  private static final String        FILENAME = "tsc.prefs";
  private static final DecimalFormat dig2     = new DecimalFormat("00");

  private static final String MODE_GT = "gametime";
  private static final String MODE_RT = "realtime";
  private static final String MODE_DN = "daynight";
  private static final List<String> MODE_LIST;

  static
  {
    MODE_LIST = new ArrayList<>(3);
    MODE_LIST.add(MODE_GT);
    MODE_LIST.add(MODE_RT);
    MODE_LIST.add(MODE_DN);
  }

  private static final String CMD_HELP  = "command_help";
  private static final String CMD_CLCK  = "command_clock";
  private static final String CMD_GAME  = "command_gametime";
  private static final String CMD_REAL  = "command_realtime";
  private static final String CMD_DAYN  = "command_daynight";
  private static final String MODE      = "mode";
  private static final String TSP_DAY   = "timespeed_day";
  private static final String TSP_NIGHT = "timespeed_night";
  private static final String CLCK_VIS  = "clock_visible";
  private static final String CLCK_X    = "clock_posx";
  private static final String CLCK_Y    = "clock_posy";
  private static final String CLCK_SIZE = "clock_size";

  private final Runnable   timerTask_FP;
  private       Properties props;
  private       Player     player;

  private float origTimeSpeed;

  private int second;
  private int minute;
  private int hour;

  public TimeSpeedChanger()
  {
    super();

    timerTask_FP = this::timerTask;
  }

  @Override
  public void onEnable()
  {
    Server server = getServer();
    // only singleplayer
    if(server.getType() != Server.Type.Singleplayer)
    {
      return;
    }
    // rescue the timespeed defined in Rising World's config.properties
    origTimeSpeed = server.getGameTimeSpeed();

    props = new Properties();

    loadPreferences();
    checkPreferences();
    // save all properties to amend corrupt or missing ones
    savePreferences();
    registerEventListener(this);
  }

  @Override
  public void onDisable()
  {

  }

  @EventMethod(Threading.Sync)
  public void onPlayerSpawn(PlayerSpawnEvent evt)
  {
    player = evt.getPlayer();

    player.sendTextMessage(
      "TimeSpeedChanger loaded, actual mode = " + props.get(MODE) + ", enter /" + props.get(CMD_HELP) + " for help page");
    player.sendTextMessage("");

    GuiLabel label = new GuiLabel(getFloat(props.getProperty(CLCK_X)), getFloat(props.getProperty(CLCK_Y)), true);
    label.setPivot(PivotPosition.TopLeft);
    label.setFont(Font.DefaultMono_Bold);
    label.setFontSize(getInteger(props.getProperty(CLCK_SIZE)));
    label.setClickable(false);
    label.setBorderColor(0.2f, 0.2f, 0.2f, 0.5f);
    label.setBorderThickness(0.6f, true);
    label.setText("          ");

    player.addGuiElement(label);
    player.setAttribute("tsc_clock_label", label);

    setGuiClockVisible();

    Timer timer = new Timer(1.0f, 0, -1, timerTask_FP);
    timer.start();

    updateRealTime();
  }

  @EventMethod(Threading.Async)
  public void onPlayerCommand(PlayerCommandEvent evt)
  {
    Server server = getServer();

    StringTokenizer st = new StringTokenizer(evt.getCommand(), " /");
    if(st.countTokens() <= 0)
    {
      return;
    }

    Properties p = props;

    String cmd = st.nextToken();

    if(cmd.equals(p.getProperty(CMD_HELP)))
    {
      printHelp();
    }
    else if(cmd.equals(p.getProperty(CMD_CLCK)))
    {
      toggleClock();
    }
    else if(cmd.equals(p.getProperty(CMD_GAME)))
    {
      setGameTime();
    }
    else if(cmd.equals(p.getProperty(CMD_REAL)))
    {
      setRealTime();
    }
    else if(cmd.equals(p.getProperty(CMD_DAYN)))
    {
      if(st.countTokens() != 2)
      {
        player.sendTextMessage("error: two numbers expected");
        return;
      }
      float day   = Float.parseFloat(p.getProperty(TSP_DAY));
      float night = Float.parseFloat(p.getProperty(TSP_NIGHT));

      try
      {
        day = Float.parseFloat(st.nextToken());
        night = Float.parseFloat(st.nextToken());
      }
      catch(NumberFormatException nfex)
      {
        player.sendTextMessage("error: not a number");
        return;
      }

      setDayNight(day, night);
    }

    // saving properties here (does not work in onDisable()?)
    savePreferences();
  }

  @EventMethod(Threading.Async)
  public void onPlayerSleep(PlayerSleepEvent evt)
  {
    if(evt.isSleeping())
    {
      return;
    }

    updateRealTime();
  }

  private void timerTask()
  {
    updateTime();

    updateLabel();

    updateDayNight();
  }

  private void updateTime()
  {
    Calendar calendar = Calendar.getInstance();

    second = calendar.get(Calendar.SECOND);
    minute = calendar.get(Calendar.MINUTE);
    hour = calendar.get(Calendar.HOUR_OF_DAY);
  }

  private void updateLabel()
  {
    if(!isClockVisible())
    {
      return;
    }

    GuiLabel label = (GuiLabel) player.getAttribute("tsc_clock_label");
    if(label == null)
    {
      return;
    }

    String time = " " + dig2.format(hour) + ":" + dig2.format(minute) + ":" + dig2.format(second) + " ";
    label.setText(time);
  }

  private void updateRealTime()
  {
    if(!MODE_RT.equals(props.getProperty(MODE)))
    {
      return;
    }

    updateTime();

    setRealTime();
  }

  private void updateDayNight()
  {
    if(!MODE_DN.equals(props.getProperty(MODE)))
    {
      return;
    }

    Server server = getServer();

    int hours = server.getGameTime().getHours();
    if(hours >= 8 && hours < 22)
    {
      server.setGameTimeSpeed(Float.parseFloat(props.getProperty(TSP_DAY)));
    }
    else
    {
      server.setGameTimeSpeed(Float.parseFloat(props.getProperty(TSP_NIGHT)));
    }
  }

  private boolean isClockVisible()
  {
    return Boolean.parseBoolean(props.getProperty(CLCK_VIS));
  }

  private void toggleClock()
  {
    props.setProperty(CLCK_VIS, Boolean.toString(!isClockVisible()));
    setGuiClockVisible();
  }

  private void setGuiClockVisible()
  {
    GuiLabel label = (GuiLabel) player.getAttribute("tsc_clock_label");

    if(label == null)
    {
      return;
    }
    player.setGuiElementVisible(label, Boolean.parseBoolean(props.getProperty(CLCK_VIS)));
  }

  private void setGameTime()
  {
    props.setProperty(MODE, MODE_GT);
    getServer().setGameTimeSpeed(origTimeSpeed);
  }

  private void setRealTime()
  {
    props.setProperty(MODE, MODE_RT);
    Server server = getServer();
    server.setGameTimeSpeed(60.0f);
    server.setGameTime(hour, minute);
  }

  private void setDayNight(final float p_day, final float p_night)
  {
    if(p_day <= 0 || p_night <= 0)
    {
      return;
    }
    props.setProperty(MODE, MODE_DN);
    props.setProperty(TSP_DAY, Float.toString(p_day));
    props.setProperty(TSP_NIGHT, Float.toString(p_night));
  }

  private void printHelp()
  {
    player.sendTextMessage("");
    player.sendTextMessage("");
    player.sendTextMessage("*************************************************************************");
    player.sendTextMessage("*                   TimeSpeedChanger Plugin - help page                 *");
    player.sendTextMessage("*************************************************************************");
    player.sendTextMessage("* ");
    player.sendTextMessage("* Available commands:");
    player.sendTextMessage("* /" + props.getProperty(CMD_HELP));
    player.sendTextMessage("*   show this help page");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_CLCK));
    player.sendTextMessage("*   show/ hide realtime clock on screen");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_GAME));
    player.sendTextMessage("*   switch to normal ingame timespeed");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_REAL));
    player.sendTextMessage("*   switch to current time and realtime timespeed");
    player.sendTextMessage("* ");
    player.sendTextMessage("* /" + props.getProperty(CMD_DAYN) + " x.xx y.yy");
    player.sendTextMessage("*   set distinct timespeed for day (08:00-22:00) and night (22:00-08:00),");
    player.sendTextMessage("*   the amount of realtime seconds for one ingame minute, f.e.:");
    player.sendTextMessage("*   /" + props.getProperty(CMD_DAYN) + " 3.0 2.0");
    player.sendTextMessage("*   a minute at daytime takes 3 seconds, at night it takes 2 seconds");
    player.sendTextMessage("* ");
    player.sendTextMessage("* The readme.txt in the plugin folder contains further information.");
    player.sendTextMessage("*************************************************************************");
    player.sendTextMessage("");
    player.sendTextMessage("(Activate chat and press <Page Up> <Page Down> to scroll) ");
    player.sendTextMessage("");
  }

  public void checkPreferences()
  {
    String str = props.getProperty(CMD_HELP);
    if(str == null || str.length() <= 0)
    {
      props.setProperty(CMD_HELP, "tschelp");
    }

    str = props.getProperty(CMD_CLCK);
    if(str == null || str.length() <= 0)
    {
      props.setProperty(CMD_CLCK, "tscclock");
    }

    str = props.getProperty(CMD_GAME);
    if(str == null || str.length() <= 0)
    {
      props.setProperty(CMD_GAME, "tscgametime");
    }

    str = props.getProperty(CMD_REAL);
    if(str == null || str.length() <= 0)
    {
      props.setProperty(CMD_REAL, "tscrealtime");
    }

    str = props.getProperty(CMD_DAYN);
    if(str == null || str.length() <= 0)
    {
      props.setProperty(CMD_DAYN, "tscdaynight");
    }

    str = props.getProperty(MODE);
    if(str == null || !MODE_LIST.contains(str))
    {
      props.setProperty(MODE, MODE_GT);
    }

    float f = getFloat(props.getProperty(TSP_DAY));
    if(f <= 0.01f || f > 999.99f)
    {
      props.setProperty(TSP_DAY, "1.75");
    }

    f = getFloat(props.getProperty(TSP_NIGHT));
    if(f <= 0.01f || f > 999.99f)
    {
      props.setProperty(TSP_NIGHT, "1.75");
    }

    f = getFloat(props.getProperty(CLCK_X));
    if(f < 0.00f || f > 1.00f)
    {
      props.setProperty(CLCK_X, "0.0");
    }

    f = getFloat(props.getProperty(CLCK_Y));
    if(f < 0.00f || f > 1.00f)
    {
      props.setProperty(CLCK_Y, "1.0");
    }

    int i = getInteger(props.getProperty(CLCK_SIZE));
    if(i < 1 || i > 100)
    {
      props.setProperty(CLCK_SIZE, "20");
    }

    str = props.getProperty(CLCK_VIS);
    if(str == null || !str.equals("true") || !str.equals("false"))
    {
      props.setProperty(CLCK_VIS, "true");
    }
  }

  public static float getFloat(String str)
  {
    if(str == null)
    {
      str = "";
    }
    float f;
    try
    {
      f = Float.parseFloat(str);
    }
    catch(NumberFormatException ex)
    {
      f = Float.POSITIVE_INFINITY;
    }

    return f;
  }

  public static int getInteger(String str)
  {
    if(str == null)
    {
      str = "";
    }
    int i;
    try
    {
      i = Integer.parseInt(str);
    }
    catch(NumberFormatException ex)
    {
      i = 0;
    }

    return i;
  }

  public void loadPreferences()
  {
    String path = getPath();
    if(path == null || path.length() < 1)
    {
      return;
    }
    String filename = path + FileSystems.getDefault().getSeparator() + FILENAME;
    File   file     = new File(filename);
    if(!file.exists())
    {
      savePreferences();
    }
    try(Reader reader = new FileReader(file))
    {
      props.load(reader);
      reader.close();
    }
    catch(IOException ex)
    {
      // no-op
    }
  }

  public void savePreferences()
  {
    String path = getPath();
    if(path == null || path.length() < 1)
    {
      return;
    }
    String filename = path + FileSystems.getDefault().getSeparator() + FILENAME;

    File file = new File(filename);
    try
    {
      boolean b = file.createNewFile();
    }
    catch(IOException ex)
    {
      // no-op
    }
    try(Writer writer = new FileWriter(file, false))
    {
      props.store(writer, null);
      writer.flush();
      writer.close();
    }
    catch(IOException ex)
    {
      // no-op
    }
  }
}
